structure HashTable =
struct
datatype ('a, 'b) hash_table = HT of {
	entries : ('a * 'b) HashSet.t ref,
	hash_fn : 'a -> word,
	eq_fn : ('a * 'a) -> bool
	}

exception InconsistentState

(* The HashTable requires functions that just work on the index
* But the underlying entries need a hash function that works
* on the key, value pair. this creates one out of the other.
*)
fun hashSet_hash_fn f (key, _(*val*)) = f key 

(* Create a new table; the int is a size hint and the exception
* is to be raised by find.
*)
fun new (sizeHint, hash_fn, eq_fn) = HT{
	entries = ref (HashSet.newOfSize {
				hash = hashSet_hash_fn hash_fn,
				size = sizeHint}),
	hash_fn = hash_fn,
	eq_fn = eq_fn
	}

fun get (HT{entries, eq_fn, hash_fn, ...}) key =
	let val entOpt = HashSet.peek(!entries, hash_fn key, fn (k, _) => eq_fn (key, k))
	in case entOpt
		of NONE => NONE
		| SOME (_(*key*),v) => SOME (v)
	end

fun put (HT{entries, eq_fn, hash_fn}) (key, value) =
	let val hashSet = !entries
	in ignore (HashSet.insertIfNew(hashSet,
					hash_fn key,
					fn (k, _) => eq_fn (key, k), 
					fn () => (key, value),
					fn ((*k, v*) _, _) => 
                                           let val _ = HashSet.remove(hashSet, hash_fn key, fn (k, _) => eq_fn (key, k))
                                               val _ = HashSet.insertIfNew(hashSet, hash_fn key, 
                                                                           fn (k, _) => eq_fn (key, k),
                                                                           fn () => (key, value),
                                                                           fn ((*k, v*) _, _) => raise InconsistentState)
                                           in ()
                                           end))
	end

fun size (HT{entries, ...}) = HashSet.size (!entries)

fun fold (HT{entries, ...}) init f = HashSet.fold (!entries, init, f)

(* Used to prepend channel nodes to the thread list to save when pruning the graph. *)
fun foldValues (HT{entries, ...}) init f = HashSet.fold (!entries, init, fn ((_(*key*), v), acc) => f (v, acc))

fun getValues table =
	foldValues table [] (fn(v, acc)=> v::acc)

fun remove (HT{entries, hash_fn, eq_fn, ...}) key =
	HashSet.remove (!entries, hash_fn key, fn (k, _) => eq_fn (key, k)) handle _ => () (*exception raised because entry was not in the table *)

fun putCond table replaceCond (key, value) =
  case (get table key)
    of NONE => put table (key, value)
     | SOME v2 => if replaceCond(value, v2)
                  then put table (key, value)
                  else ()

end