// Take a look at the license at the top of the repository in the LICENSE file.

// rustdoc-stripper-ignore-next
//! Traits intended for subclassing [`GLArea`].

use gdk::GLContext;
use glib::translate::*;

use crate::{GLArea, ffi, prelude::*, subclass::prelude::*};

#[allow(clippy::upper_case_acronyms)]
pub trait GLAreaImpl: WidgetImpl + ObjectSubclass<Type: IsA<GLArea>> {
    fn create_context(&self) -> Option<GLContext> {
        self.parent_create_context()
    }

    fn render(&self, context: &GLContext) -> glib::Propagation {
        self.parent_render(context)
    }

    fn resize(&self, width: i32, height: i32) {
        self.parent_resize(width, height)
    }
}

#[allow(clippy::upper_case_acronyms)]
pub trait GLAreaImplExt: GLAreaImpl {
    fn parent_create_context(&self) -> Option<GLContext> {
        unsafe {
            let data = Self::type_data();
            let parent_class = data.as_ref().parent_class() as *mut ffi::GtkGLAreaClass;
            if let Some(f) = (*parent_class).create_context {
                return from_glib_none(f(self.obj().unsafe_cast_ref::<GLArea>().to_glib_none().0));
            };
            None
        }
    }

    fn parent_render(&self, context: &GLContext) -> glib::Propagation {
        unsafe {
            let data = Self::type_data();
            let parent_class = data.as_ref().parent_class() as *mut ffi::GtkGLAreaClass;
            let f = (*parent_class)
                .render
                .expect("No parent class impl for \"render\"");
            from_glib(f(
                self.obj().unsafe_cast_ref::<GLArea>().to_glib_none().0,
                context.to_glib_none().0,
            ))
        }
    }

    fn parent_resize(&self, width: i32, height: i32) {
        unsafe {
            let data = Self::type_data();
            let parent_class = data.as_ref().parent_class() as *mut ffi::GtkGLAreaClass;
            if let Some(f) = (*parent_class).resize {
                f(
                    self.obj().unsafe_cast_ref::<GLArea>().to_glib_none().0,
                    width,
                    height,
                )
            }
        }
    }
}

impl<T: GLAreaImpl> GLAreaImplExt for T {}

unsafe impl<T: GLAreaImpl> IsSubclassable<T> for GLArea {
    fn class_init(class: &mut glib::Class<Self>) {
        Self::parent_class_init::<T>(class);

        let klass = class.as_mut();
        klass.create_context = Some(gl_area_create_context::<T>);
        klass.render = Some(gl_area_render::<T>);
        klass.resize = Some(gl_area_resize::<T>);
    }
}

unsafe extern "C" fn gl_area_create_context<T: GLAreaImpl>(
    ptr: *mut ffi::GtkGLArea,
) -> *mut gdk::ffi::GdkGLContext {
    unsafe {
        let instance = &*(ptr as *mut T::Instance);
        let imp = instance.imp();

        imp.create_context().into_glib_ptr()
    }
}

unsafe extern "C" fn gl_area_render<T: GLAreaImpl>(
    ptr: *mut ffi::GtkGLArea,
    context: *mut gdk::ffi::GdkGLContext,
) -> glib::ffi::gboolean {
    unsafe {
        let instance = &*(ptr as *mut T::Instance);
        let imp = instance.imp();

        imp.render(&from_glib_borrow(context)).into_glib()
    }
}

unsafe extern "C" fn gl_area_resize<T: GLAreaImpl>(
    ptr: *mut ffi::GtkGLArea,
    width: i32,
    height: i32,
) {
    unsafe {
        let instance = &*(ptr as *mut T::Instance);
        let imp = instance.imp();

        imp.resize(width, height)
    }
}
